using System;
using System.Xml;
using gov.va.med.vbecs.DAL.VistALink.OpenLibrary.Messages;

namespace gov.va.med.vbecs.DAL.VistALink.OpenLibrary
{
	/// <summary>
	/// This class accepts XML string, parses it, analyzes it 
	/// and creates appropriate VistALink message.
	/// </summary>
	public class VistALinkMessageXmlFactory
	{
		RpcParameterXmlFactory _rpcParameterFactory;

		/// <summary>
		/// Constructor accepting RpcParameterXmlFactory as parameter 
		/// (required to recognize and create RpcRequestMessage).
		/// The only public method of this class is made instance virtual method, so
		/// it may be overriden by derived classes in the future.
		/// </summary>
		/// <param name="rpcParameterFactory">
		///		RpcParameterXmlFactory needed to recognize and create RpcRequestMessage.
		///	</param>
		public VistALinkMessageXmlFactory( RpcParameterXmlFactory rpcParameterFactory ) 
		{
			if( rpcParameterFactory == null )
				throw( new ArgumentNullException( "rpcParameterFactory" ) );

			_rpcParameterFactory = rpcParameterFactory;
		}

		/// <summary>
		/// Parses supplied XML string, determines if it contains known VistALink message
		/// XML and returns fully initialized instance of appropriate VistALink 
		/// message deserialized from a given string. 
		/// It will return null if XML is not recognized. 
		/// </summary>
		/// <param name="xmlString">Source XML string.</param>
		/// <returns>
		///		Instance of VistALink message deserialized from 
		///		source XML string or null if XML is not recognized.
		/// </returns>
		public virtual VistALinkMessage ParseXmlCreateVistALinkMessage( string xmlString )
		{
			if( xmlString == null )
				throw( new ArgumentNullException( "xmlString" ) );


			XmlDocument _doc = new XmlDocument();

            _doc.PreserveWhitespace = false;

			try
			{
                //CR 3518
			    _doc.XmlResolver = null;
				_doc.LoadXml( xmlString );
			}
			catch( XmlException xcp )
			{
				throw( new MalformedVistALinkMessageXmlSourceException( xcp ) );
			}

			switch( VistALinkMessage.XmlParseGetVistALinkMessageTypeString( _doc ) )
			{
				case VistALinkSimpleRequestMessage.MessageType:
					return ParseGetVistALinkRequestMessage( _doc );

				case VistALinkSimpleResponseMessage.MessageType:
					return ParseGetVistALinkResponseMessage( _doc );

				case VistALinkFaultMessage.MessageType:
					return new VistALinkFaultMessage( _doc );				

				case SecurityBaseRequestMessage.MessageType:
					return ParseGetSecurityRequestMessage( _doc );

				case SecurityBaseResponseMessage.MessageType:
					return ParseGetSecurityResponseMessage( _doc );

				case SecurityFaultMessage.MessageType:
					return new SecurityFaultMessage( _doc );

				case RpcRequestMessage.MessageType:
					return new RpcRequestMessage( _doc, _rpcParameterFactory );

				case RpcResponseMessage.MessageType:
					return new RpcResponseMessage( _doc );
				
				case RpcFaultMessage.MessageType:
					return new RpcFaultMessage( _doc );
			}

			return null;
		}

		/// <summary>
		/// The method analyses a given XML document and returns matching instance of 
		/// VistALink security response message deserialized from supplied document.
		/// </summary>
		/// <param name="sourceDoc">XML document to use.</param>
		/// <returns>
		///		New instance of VistALink security response message 
		///		deserialized from a given XML document.
		///	</returns>
		private SecurityBaseResponseMessage ParseGetSecurityResponseMessage( XmlDocument sourceDoc )
		{
			string _responseType = SecurityBaseResponseMessage.XmlParseGetResponseTypeString( sourceDoc );
			switch( _responseType )
			{
				case SecurityGetUserDemographicsResponseMessage.ResponseType:
					return new SecurityGetUserDemographicsResponseMessage( sourceDoc );

				case SecuritySetupAndIntroResponseMessage.ResponseType:
					return new SecuritySetupAndIntroResponseMessage( sourceDoc );

				case SecuritySelectDivisionResponseMessage.ResponseType:
					return new SecuritySelectDivisionResponseMessage( sourceDoc );

				case SecurityLogoutResponseMessage.ResponseType:
					return new SecurityLogoutResponseMessage( sourceDoc );

				default:
					// Handle special cases (partial success responses possible)
					if( IsLogonResponseMessageType( _responseType ) )
						return ParseGetSecurityLogonResponseMessage( sourceDoc );

					if( IsUpdateVerifyCodeResponseMessageType( _responseType ) )
						return ParseGetSecurityUpdateVerifyCodeResponseMessage( sourceDoc );

					return null;
			}			
		}

		/// <summary>
		/// The method checks supplied response type to find out if it matches any of 
		/// types appropriate for security logon response messages
		/// (Currently all these messages share the same response type. 
		/// However they may change. This is the safety valve.).
		/// </summary>
		/// <param name="responseType">Response type to check.</param>
		/// <returns>True if a given response type is matches any of logon response message types.</returns>
		private bool IsLogonResponseMessageType( string responseType )
		{
			return 
				( responseType == SecurityLogonDivisionRequiredResponseMessage.ResponseType ) ||
				( responseType == SecurityLogonVerifyCodeChangeRequiredResponseMessage.ResponseType ) ||
				( responseType == SecurityLogonSuccessResponseMessage.ResponseType ) ||
				( responseType == SecurityLogonFailureResponseMessage.ResponseType );
		}

		/// <summary>
		/// The method analyses a given XML document and returns matching instance of 
		/// VistALink security request message deserialized from supplied document.
		/// </summary>
		/// <param name="sourceDoc">XML document to use.</param>
		/// <returns>
		///		New instance of VistALink security request message 
		///		deserialized from a given XML document.
		///	</returns>
		private SecurityBaseResponseMessage ParseGetSecurityLogonResponseMessage( XmlDocument sourceDoc )
		{
			ResponseStatus _msgStatus = BaseResponseMessage.XmlParseGetResponseStatus( sourceDoc );

			if( _msgStatus == ResponseStatus.Success )
				return new SecurityLogonSuccessResponseMessage( sourceDoc );

			if( _msgStatus == ResponseStatus.Failure )
				return new SecurityLogonFailureResponseMessage( sourceDoc );

			if( _msgStatus != ResponseStatus.PartialSuccess )
				return null;

			if( SecurityLogonVerifyCodeChangeRequiredResponseMessage.XmlParseDoesPartialSuccessXmlMatchMessage( sourceDoc ) )
				return new SecurityLogonVerifyCodeChangeRequiredResponseMessage( sourceDoc );

			if( SecurityLogonDivisionRequiredResponseMessage.XmlParseDoesPartialSuccessXmlHaveDivisionsData( sourceDoc ) )
				return new SecurityLogonDivisionRequiredResponseMessage( sourceDoc );

			return null;
		}

		/// <summary>
		/// The method analyses a given XML document and returns matching instance of 
		/// security update verify code response message deserialized from supplied document.
		/// </summary>
		/// <param name="sourceDoc">XML document to use.</param>
		/// <returns>
		///		New instance of security update verify code response message
		///		deserialized from a given XML document.
		///	</returns>
		private SecurityBaseResponseMessage ParseGetSecurityUpdateVerifyCodeResponseMessage( XmlDocument sourceDoc )
		{			
			ResponseStatus _msgStatus = BaseResponseMessage.XmlParseGetResponseStatus( sourceDoc );

			if( _msgStatus == ResponseStatus.PartialSuccess )
				return new SecurityUpdateVerifyCodeDivisionRequiredResponseMessage( sourceDoc );
			
			if( _msgStatus == ResponseStatus.Failure || _msgStatus == ResponseStatus.Success )
				return new SecurityUpdateVerifyCodeResponseMessage( sourceDoc );

			return null;
		}

		/// <summary>
		/// The method checks supplied response type to find out if it matches any of 
		/// types appropriate for security update verify code response messages
		/// (Currently all these messages share the same response type. 
		/// However they may change. This is the safety valve.).
		/// </summary>
		/// <param name="responseType">Response type to check.</param>
		/// <returns>
		///		True if a given response type is matches any of security 
		///		update verify code response message types.
		///	</returns>
		private bool IsUpdateVerifyCodeResponseMessageType( string responseType )
		{
			return
				( responseType == SecurityUpdateVerifyCodeDivisionRequiredResponseMessage.ResponseType ) ||
				( responseType == SecurityUpdateVerifyCodeResponseMessage.ResponseType );
		}

		/// <summary>
		/// The method analyses a given XML document and returns matching instance of 
		/// VistALink security request message deserialized from supplied document.
		/// </summary>
		/// <param name="sourceDoc">XML document to use.</param>
		/// <returns>
		///		New instance of VistALink security request message 
		///		deserialized from a given XML document.
		///	</returns>
		private SecurityBaseRequestMessage ParseGetSecurityRequestMessage( XmlDocument sourceDoc )
		{
			switch( SecurityBaseRequestMessage.XmlParseGetRequestTypeString( sourceDoc ) )
			{
				case SecurityGetUserDemographicsRequestMessage.RequestType:
					return new SecurityGetUserDemographicsRequestMessage( sourceDoc );

				case SecuritySetupAndIntroRequestMessage.RequestType:
					return new SecuritySetupAndIntroRequestMessage( sourceDoc );

				case SecuritySelectDivisionRequestMessage.RequestType:
					return new SecuritySelectDivisionRequestMessage( sourceDoc );

				case SecurityLogonRequestMessage.RequestType:
					return new SecurityLogonRequestMessage( sourceDoc );

				case SecurityLogoutRequestMessage.RequestType:
					return new SecurityLogoutRequestMessage( sourceDoc );

				case SecurityUpdateVerifyCodeRequestMessage.RequestType:
					return new SecurityUpdateVerifyCodeRequestMessage( sourceDoc );

				default:
					return null;
			}			
		}

		/// <summary>
		/// The method analyses a given XML document and returns matching instance of 
		/// VistALink simple response message deserialized from supplied document.
		/// </summary>
		/// <param name="sourceDoc">XML document to use.</param>
		/// <returns>
		///		New instance of VistALink simple response message 
		///		deserialized from a given XML document.
		///	</returns>
		private VistALinkSimpleResponseMessage ParseGetVistALinkResponseMessage( XmlDocument sourceDoc )
		{
			switch( BaseResponseMessage.XmlParseGetResponseTypeString( sourceDoc ) )
			{
				case VistALinkHeartbeatResponseMessage.ResponseType:
					return new VistALinkHeartbeatResponseMessage( sourceDoc );

				case VistALinkCloseSocketResponseMessage.ResponseType:
					return new VistALinkCloseSocketResponseMessage( sourceDoc );

				default:
					return null;
			}
		}

		/// <summary>
		/// The method analyses a given XML document and returns matching instance of 
		/// VistALink simple request message deserialized from supplied document.
		/// </summary>
		/// <param name="sourceDoc">XML document to use.</param>
		/// <returns>
		///		New instance of VistALink simple request message 
		///		deserialized from a given XML document.
		///	</returns>
		private VistALinkSimpleRequestMessage ParseGetVistALinkRequestMessage( XmlDocument sourceDoc )
		{
			switch( BaseRequestMessage.XmlParseGetRequestTypeString( sourceDoc ) )
			{
				case VistALinkHeartbeatRequestMessage.RequestType:
					return new VistALinkHeartbeatRequestMessage( sourceDoc );

				case VistALinkCloseSocketRequestMessage.RequestType:
					return new VistALinkCloseSocketRequestMessage( sourceDoc );

				default:
					return null;
			}
		}
	}
}